import React, { Component } from 'react';
import { shell } from 'electron';
import { History, Location } from 'history';
import 'react-circular-progressbar/dist/styles.css';
import { withStyles } from '@material-ui/core/styles';
import Loading from 'components/Loading';
import HeaderLabel from 'components/HeaderLabel';
import { Logger } from 'library/Logger';
import {
    checkForUpdates,
    downloadUpdate,
    quitAndInstallUpdate
} from 'library/AutoUpdater';
import { OS_IS_WINDOWS } from 'constants/versions';
import { SUPPORT_URL } from 'constants/urls';
import routes from 'constants/routes';
import { MODELS, HEADSET_STATUS } from 'library/headset/consts';
import DownloadingUpdate from './DownloadingUpdate';
import NewUpdateAvailable from './NewUpdateAvailable';
import DownloadingUpdateComplete from './DownloadingUpdateComplete';
import NoDeviceConnected from './NoDeviceConnected';
import FWUpdateInterruptedModal from './FWUpdateInterruptedModal';
import DownloadFailedScreen from './DownloadFailedScreen';
import styles from './styles';
import DeviceConnecting from './DeviceConnecting';
import NoInternet from '../UpdateDeviceScreen/NoInternet';
import ManualDownloadScreen from './ManualDownloadScreen';

type Props = {
    device: object,
    language: object,
    strings: object,
    location: Location,
    history: History,
    classes: {
        [key: string]: string
    }
};

let onlineCheckTimeout;
const CHECK_NETWORK_TIMEOUT = 2000;

class WelcomeScreen extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            loading: true,
            updateAvailable: false,
            downloadingUpdate: false,
            appDownloadProgress: 0,
            downloadingUpdateComplete: false,
            waitingForDevice: false,
            updateError: false,
            device: props.device,
            language: props.language,
            strings: props.strings,
            skipAutoUpdate: props.location.state,
            noInternet: false,
            checkForConnectivityTimer: null,
            manualAutoUpdate: false
        };
    }

    componentDidMount() {
        Logger.info('WelcomeScreen: Mounted');

        const { skipAutoUpdate } = this.state;

        if (!skipAutoUpdate) {
            // Check for new Cardo Update app version
            Logger.info('WelcomeScreen: Checking for app updates');
            checkForUpdates(this);
        } else {
            // Skip auto update
            this.autoUpdateNotAvailable();
        }

        window.addEventListener('online', () => this.updateOnlineStatus());
        window.addEventListener('offline', () => this.updateOnlineStatus());
        this.setState({
            checkForConnectivityTimer: setInterval(() => {
                this.updateOnlineStatus();
            }, 1000)
        });
    }

    componentDidUpdate(prevProps, prevState) {
        const { device, strings, language } = this.props;
        Logger.info(
            'componentDidUpdate - prevDevice',
            prevState.device,
            'newDevice',
            device
        );

        if (
            (device.notSupported && !prevState.device.notSupported) ||
            device.model === MODELS.PRO1
        ) {
            // Device not supported
            Logger.info(
                'componentDidUpdate - Not updating device - unsupported device found - ',
                prevState.device,
                device
            );
            this.moveToUnsupportedDevice();
        } else if (prevState.device.isDfu !== device.isDfu) {
            this.setState({ device });
        } else if (
            prevState.device.connected !== device.connected ||
            (device.connected === HEADSET_STATUS.HEADSET_GETTING_CONNECTED &&
                prevState.device.progress !== device.progress)
        ) {
            // Device got connected
            Logger.info(
                'WelcomeScreen: componentDidUpdate - updating device - ',
                prevState.connected,
                device.connected,
                device
            );
            this.setState({ device });
            if (device.fwError) {
                Logger.info(
                    'WelcomeScreen: componentDidUpdate - calling moveToUSBErrorMessage',
                    prevState.connected,
                    device
                );
                this.moveToUSBErrorMessage();
            } else if (
                prevState.waitingForDevice &&
                device.connected === HEADSET_STATUS.HEADSET_CONNECTED
            ) {
                // We're currently waiting for device and device is already connected - move to update device screen
                Logger.info(
                    'WelcomeScreen: componentDidUpdate - calling moveToDeviceUpdateScreen - ',
                    prevState.connected,
                    device.connected,
                    prevState.waitingForDevice
                );
                this.moveToDeviceUpdateScreen();
            }
        } else if (device.fwError) {
            Logger.info(
                'WelcomeScreen: componentDidUpdate 2 - calling moveToUSBErrorMessage',
                prevState.connected,
                device
            );
            this.moveToUSBErrorMessage();
        }

        if (prevState.language !== language) {
            this.setState({ language, strings });
        }
    }

    componentWillUnmount() {
        const { checkForConnectivityTimer } = this.state;

        window.removeEventListener('online', () => this.updateOnlineStatus());
        window.removeEventListener('offline', () => this.updateOnlineStatus());
        onlineCheckTimeout = undefined;
        clearTimeout(onlineCheckTimeout);
        clearTimeout(checkForConnectivityTimer);
    }

    updateOnlineStatus() {
        const { downloadingUpdate, updateAvailable } = this.state;

        if (!navigator.onLine) {
            console.log('updateOnlineStatus - navigator not onLine');

            this.setState({
                loading: false,
                updateAvailable: false,
                downloadingUpdate: false,
                downloadingUpdateComplete: false,
                waitingForDevice: true,
                updateError: downloadingUpdate,
                noInternet: !downloadingUpdate
            });
        } else if (downloadingUpdate) {
            this.setState({
                loading: false,
                updateAvailable: false,
                downloadingUpdate: true,
                downloadingUpdateComplete: false,
                waitingForDevice: false,
                updateError: false
            });
        } else if (updateAvailable) {
            this.setState({
                loading: false,
                updateAvailable: true,
                updateError: false
            });
        }
    }

    checkOnlineStatus() {
        onlineCheckTimeout = setTimeout(() => {
            const {
                downloadingUpdateComplete,
                updateError,
                downloadingUpdate
            } = this.state;
            if (
                downloadingUpdate &&
                !updateError &&
                !downloadingUpdateComplete &&
                navigator.onLine
            ) {
                Logger.info('Checking status in timeout');
                this.checkOnlineStatus();
            } else if (!navigator.onLine) {
                Logger.info('Timeout is not online');
                this.setState({
                    loading: false,
                    updateAvailable: false,
                    downloadingUpdate: false,
                    downloadingUpdateComplete: false,
                    waitingForDevice: true,
                    updateError: true
                });
            }
        }, CHECK_NETWORK_TIMEOUT);
    }

    onNeedSupport() {
        shell.openExternal(SUPPORT_URL);
    }

    onInstallUpdate() {
        quitAndInstallUpdate();
    }

    onDownloadUpdate() {
        // Only for Windows for NSIS there is an issue to trigger offline mode
        if (OS_IS_WINDOWS) {
            this.checkOnlineStatus();
        }

        // Initiate app update download
        this.setState({ updateAvailable: false, downloadingUpdate: true });

        downloadUpdate();
    }

    onRetryClick() {
        this.setState(
            {
                loading: true,
                updateAvailable: false,
                downloadingUpdate: false,
                downloadingUpdateComplete: false,
                waitingForDevice: false,
                updateError: false
            },
            () => {
                checkForUpdates(this);
            }
        );
    }

    autoUpdateError(_error) {
        // Move to waiting-for-device state
        Logger.error('autoUpdateError', _error);
        this.setState({
            loading: false,
            updateAvailable: false,
            downloadingUpdate: false,
            downloadingUpdateComplete: false,
            waitingForDevice: true,
            // Auto update failed because we're not connected to the Internet
            updateError: false,
            manualAutoUpdate: navigator.onLine
        });
    }

    autoUpdateAvailable(_info) {
        this.setState({
            loading: false,
            updateAvailable: true,
            updateError: false
        });
    }

    autoUpdateNotAvailable(_info) {
        const { device } = this.state;

        // Move to check-for-device state
        this.setState({ loading: false, waitingForDevice: true });

        if (
            device.connected === HEADSET_STATUS.HEADSET_CONNECTED &&
            !device.isDfu &&
            !device.fwError
        ) {
            // We're currently waiting for device and device is already connected - move to update device screen
            Logger.info(
                'WelcomeScreen: autoUpdateNotAvailable - calling moveToDeviceUpdateScreen ',
                device.isDfu,
                device
            );
            this.moveToDeviceUpdateScreen();
        }
    }

    autoUpdateDownloaded(_info) {
        onlineCheckTimeout = undefined;
        clearTimeout(onlineCheckTimeout);

        this.setState({
            downloadingUpdate: false,
            downloadingUpdateComplete: true,
            updateError: false
        });
    }

    autoUpdateDownloadProgress(info) {
        this.setState({
            appDownloadProgress: info.percent,
            updateError: false
        });
    }

    moveToDeviceUpdateScreen() {
        const { history } = this.props;
        Logger.info('WelcomeScreen: moveToDeviceUpdateScreen');

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(routes.UPDATE_DEVICE.path);
        }, 100);
    }

    moveToUSBErrorMessage() {
        const { history } = this.props;
        const { device } = this.state;

        Logger.info('Device with FW USB error found', device);

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(routes.USB_ERROR_UNIT.path);
        }, 100);
    }

    moveToUnsupportedDevice() {
        const { history } = this.props;
        const { device } = this.state;

        Logger.info('Unsupported device found', device);

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(routes.UNSUPPORTED_UNIT.path);
        }, 100);
    }

    renderContent() {
        const {
            strings,
            loading,
            updateAvailable,
            downloadingUpdate,
            appDownloadProgress,
            downloadingUpdateComplete,
            updateError,
            noInternet,
            manualAutoUpdate
        } = this.state;
        const { device } = this.state;

        Logger.info(
            'WelcomeScreen: renderContent - loading',
            loading,
            'fwUpdateInterrupted',
            device.isDfu,
            'device',
            device
        );

        if (loading) {
            return <Loading />;
        }
        if (updateAvailable) {
            return (
                <NewUpdateAvailable
                    strings={strings}
                    onDownloadUpdate={() => this.onDownloadUpdate()}
                />
            );
        }
        if (noInternet) {
            return (
                <NoInternet
                    strings={strings}
                    checkForUpdates={() => {
                        this.setState({ noInternet: !navigator.onLine });
                        this.onRetryClick();
                    }}
                    onNeedSupport={() => this.onNeedSupport()}
                />
            );
        }
        if (manualAutoUpdate) {
            return <ManualDownloadScreen strings={strings} />;
        }
        if (updateError) {
            return (
                <DownloadFailedScreen
                    strings={strings}
                    onRetry={() => this.onRetryClick()}
                />
            );
        }
        if (downloadingUpdate) {
            return (
                <DownloadingUpdate
                    downloadProgress={appDownloadProgress}
                    strings={strings}
                />
            );
        }
        if (downloadingUpdateComplete) {
            return (
                <DownloadingUpdateComplete
                    strings={strings}
                    onInstallUpdate={() => this.onInstallUpdate()}
                />
            );
        }
        if (
            device.connected === HEADSET_STATUS.HEADSET_DISCONNECTED ||
            device.isDfu
        ) {
            return (
                <NoDeviceConnected
                    strings={strings}
                    onNeedSupport={() => this.onNeedSupport()}
                />
            );
        }
        if (device.connected === HEADSET_STATUS.HEADSET_GETTING_CONNECTED) {
            return (
                <DeviceConnecting
                    strings={strings}
                    progress={device.progress}
                />
            );
        }
        // connected == HEADSET_STATUS.HEADSET_CONNECTED

        if (!device.fwError) {
            Logger.info(
                'WelcomeScreen: renderContent - calling moveToDeviceUpdateScreen - ',
                device.connected,
                device
            );

            // We're currently waiting for device and device is already connected - move to update device screen
            return this.moveToDeviceUpdateScreen();
        }
    }

    render() {
        const { strings, language, device, noInternet } = this.state;
        const { classes } = this.props;

        Logger.info(
            'WelcomeScreen: render - fwUpdateInterrupted',
            device.isDfu,
            'device',
            device
        );

        return (
            <div className={classes.container}>
                {!noInternet && (
                    <>
                        <HeaderLabel
                            className={classes.header}
                            strings={strings}
                            language={language}
                        >
                            {strings.welcome_title}
                        </HeaderLabel>
                        <div className={classes.subtitle}>
                            {strings.welcome_subtitle}
                        </div>
                    </>
                )}

                <div className={classes.innerContent}>
                    {this.renderContent()}
                </div>

                <FWUpdateInterruptedModal
                    open={
                        device.isDfu &&
                        device.connected !==
                            HEADSET_STATUS.HEADSET_DISCONNECTED &&
                        !noInternet
                    }
                    strings={strings}
                    btnCallback={() => {
                        Logger.info(
                            'WelcomeScreen: btnCallback - calling moveToDeviceUpdateScreen'
                        );
                        this.moveToDeviceUpdateScreen();
                    }}
                />
            </div>
        );
    }
}

export default withStyles(styles)(WelcomeScreen);
